/** \class "RPair"

 Implements a Lisp pair.

 A Lisp pair has two parts: the "car" and the "cdr".
 The "car" is the first word, and the "cdr" is the last.

 In a proper list, the "cdr" is another proper list,
  or an empty pair, "nil", signalling the end.

 A Lisp sentence is a proper list.
 A sentence is a list of words or frases separated by spaces and
 enclosed by parentheses. A frase is just a sentence inside a sentence.

 I use ``","'' instead of ``"."'' for improper lists.
 Thus, "(1 , 2)" is a pair; "1" is the "car", and "2" is the "cdr".
 In proper lists, as "(1 2)", "1" is the "car", and "(2)" is the "cdr".
 And in "(2)" the "car" is "2" and the "cdr" is "()".
 So "(1 2)" is the same as "(1 , (2 , ()))".

@author  Ramn Casares 2003
@version 2003.02.05
*/
package RLisp;

import java.util.StringTokenizer;
import java.util.Vector;

public class RPair {

 /**\variable"car"*/
 public Object car;
 /**\variable"cdr"*/
 public Object cdr;

 /**\constructor"RPair()" produces a "()" list */
 public RPair() { car = null; cdr = null; }

 /**\variable"nil" is the "()" list */
 public static final RPair nil = new RPair();

 /**\method"isNil()" checks if this "RPair" is the "nil" list */
 public boolean isNil() { return( car == null && cdr == null); }

 /**\method"isNil(Object)" checks if "o" is the "nil" list */
 public static boolean isNil(Object o) {
  if ( o instanceof RPair )
   return( ((RPair)o).car == null && ((RPair)o).cdr == null );
  else return(false);
 }


 /**\constructor"RPair(Object, Object)" */
 public RPair(Object car, Object cdr) {
  this.car = car; this.cdr = cdr;
 }

 /**\constructor"RPair(String)"*/
 public RPair(String s) {
  Object o = tokenize(s);
  if ( isRPair(o) ) {
   this.car = ((RPair)o).car;
   this.cdr = ((RPair)o).cdr;
  } else {
   this.car = "ERROR!";
   this.cdr = o;
  }
 }

 /**\variable"delims"*/
 static final String delims = "( ';,\n\r\t_)";

 /**\method"Tokenize(String)"

 It exhausts the "String" returning an "array" with the objects.
 */
 public static Object[] Tokenize(String s) {
  Vector V = new Vector();
  StringTokenizer T = new StringTokenizer(s,delims,true);
  while( T.hasMoreTokens() ) V.add(tokenize(T, null));
  return(V.toArray());
 }

 /**\method"tokenize(String)"

 It returns one "Object" only.
 It uses three "private" methods.
 */
 public static Object tokenize(String s) {
  StringTokenizer T = new StringTokenizer(s,delims,true);
  return(tokenize(T, null));
 }
 private static Object tokenize(StringTokenizer T, RPair v) {
  String t;
  while ( T.hasMoreTokens() ) { t = T.nextToken();
   if ( " ".equals(t) || "\n".equals(t) || "_".equals(t) ||
       "\r".equals(t) || "\t".equals(t) ) {}
   else if ( "(".equals(t) ) {
    if (v==null) return(tokenize(T, new RPair()));
    else          v.add(tokenize(T, new RPair()));
   }
   else if ( ")".equals(t) ) { return(v); }
   else if ( ",".equals(t) ) { v.add(tokenize(T,null), false); }
   else if ( ";".equals(t) ) {
    while( T.hasMoreTokens() ) { t = T.nextToken();
     if ( "\n".equals(t) || "\r".equals(t) ) break;
    }
   }
   else if ( "\'".equals(t) ) {
    Object arg = tokenize(T,null);
    if (v==null) return(new RPair("quote",new RPair(arg,null)));
    else          v.add(new RPair("quote",new RPair(arg,null)));
   }
   else if (v==null) return(t); else v.add(t);
  }
  return(v);
 }
 private Object add(Object last){
  if (last == null) return(null);
  else if (isNil()) { car = last; return(last); }
  else if (cdr == null) { cdr = new RPair(last,null); return(last); }
  else if (cdr instanceof RPair) return( ((RPair)cdr).add(last) );
  else return(null);
 }
 private Object add(Object last, boolean properly){
  if (properly) return(add(last));
  else if (isNil(last)) return(last);
  else if (cdr == null) { cdr = last; return(last); }
  else if (cdr instanceof RPair) return( ((RPair)cdr).add(last,false) );
  else return(null);
 }

 /**\method"isRPair(Object)" */
 public static boolean isRPair(Object o) {
  return(o instanceof RPair);
 }

 /**\method"isAtom(Object)" */
 public static boolean isAtom(Object o) {
  return( isNil(o) || !(o instanceof RPair) );
 }

 /**\method"isList(Object)"

 If the "Object o" is a proper List, it returns the length of the List.
 Otherwise it returns -1.
 */
 public static int isList(Object o) {
  if(o == null) return(-1);
  if ( o instanceof RPair ) {
   RPair p = (RPair)o;
   if ( p.car == null )
    if ( p.cdr == null ) return(0); else return(-1);
   else
    if ( p.cdr == null ) return(1); else {
     int l = isList(p.cdr);
     if (l<0) return(l); else return(1+l);
    }
  } else return(-1);
 }


 /**\method"toArray()" */
 public Object[] toArray() {
  int l = isList(this);
  if ( l >= 0 ) {
   Object[] a = new Object[l];
   for(int i=0; i<l; i++) a[i] = nth(i);
   return(a);
  } else return(null);
 }

 /**\method"toArray(Object)" */
 public static Object[] toArray(Object o) {
  int l = isList(o);
  if ( l >= 0 ) {
   Object[] a = new Object[l];
   for(int i=0; i<l; i++) a[i] = nth(o,i);
   return(a);
  } else return(null);
 }


 /**\method"cons(Object, RPair)" */
 public static RPair cons(Object car, RPair cdr) {
  if ( isNil(cdr) ) return( new RPair(car,null) );
  else              return( new RPair(car,cdr) );
 }

 /**\method"car()"*/
 public Object car() { return( car ); }
 /**\method"cdr()"*/
 public Object cdr() { return( cdr ); }

 /**\method"Cdr()"*/
 public Object Cdr() {
  if (cdr == null) return(nil); else return(cdr);
 }

 /**\method"CDR()"*/
 public RPair CDR() {
  if (cdr == null) return(nil);
  if (cdr instanceof RPair) return( ((RPair)cdr) );
  else return(null);
 }

 /**\method"nth(int)"

 @param n is the position (car is 0)
 @return the object in the "n" position
 */
 public Object nth(int n) {
  if (n < 0) return(null);
  else if (n == 0) return(car);
  else if (cdr instanceof RPair) return( ((RPair)cdr).nth(n-1) );
  else return(null);
 }

 /**\method"nth(Object, int)"*/
 public static Object nth(Object o, int n) {
  if ( o == null || n < 0 ) return(null);
  if (o instanceof RPair)
   if (n == 0) return(((RPair)o).car);
   else return( nth(((RPair)o).cdr,n-1) );
  else return(null);
 }

 /**\method"substitute(Object, Object)"*/
 public RPair substitute(Object oldo, Object newo) {
  Object newcar = null;
  if (car == null) { if (oldo == null) newcar = newo; }
  else {
   if ( car.equals(oldo) ) newcar = newo;
   else if (car instanceof RPair)
    newcar = ((RPair)car).substitute(oldo,newo);
   else newcar = car;
  }
  Object newcdr = null;
  if (cdr == null) { if (oldo == null) newcdr = newo; }
  else {
   if ( cdr.equals(oldo) ) newcdr = newo;
   else if (cdr instanceof RPair)
    newcdr = ((RPair)cdr).substitute(oldo,newo);
   else newcdr = cdr;
  }
  return(new RPair(newcar,newcdr));
 }

 private boolean loop = false;

 /**\method"toString()"*/
 public String toString() {
  if (loop) return("RPair"+hashCode());
  else {
   loop = true;
   String s = "(" + toStringWOP() + ")";
   loop = false;
   return(s);
  }
 }
 private String toStringWOP() {
  String s;
  if ( car == null ) s = "";
  else               s = car.toString();
  if ( cdr == null ) return(s);
  else if (cdr instanceof RPair) return(s+" "+((RPair)cdr).toStringWOP());
  else return( s + " , " + cdr.toString() );
 }

 /**\method"toString(boolean)"*/
 public String toString(boolean wp) {
  if (wp) return(toString());
  else return(toStringWOP());
 }

 /**\method"equals(Object)"*/
 public boolean equals(Object o) {
  if ( o == null )  return( this.car == null && this.cdr == null );
  else if (o instanceof RPair) {
   RPair p = (RPair)o;
   return ( (  p.car == this.car ||
             ( p.car != null && p.car.equals(this.car)) ) &&
            (  p.cdr == this.cdr ||
             ( p.cdr != null && p.cdr.equals(this.cdr)) ) ); }
  else return(false);
 }

}
