/** \class "RLispJava"

 It extends \See"RLisp" with the following special forms:
 {\medskipamount=0pt
 \point "(string esto es todo)"
 \point "(new Class (arg0 arg1))" with "arg" = "ob | (cons 'Class ob)"
 \point "(method [ Class | ob ] Method (arg0 arg1))"
 \point "(array Class (ob0 ob1 ob2))"
 \point "(field [ Class | ob ] Field [ val | ])"
 \point "(path URL)"
 \point "(load URL)"
 }

@author  Ramn Casares 2003
@version 2003.02.08
*/
package RLisp;

import java.lang.reflect.Constructor;
import java.lang.reflect.Method;
import java.lang.reflect.Array;
import java.lang.reflect.Field;
import java.lang.reflect.InvocationTargetException;
import java.net.URL;
import java.io.InputStreamReader;
import java.io.File;
import java.io.FileReader;
import java.io.BufferedReader;

public class RLispJava extends RLisp {

 /**\variable"rcl" is the incremental class loader used */
 private RClassLoader rcl;

 /**\variable"baseURL" is the base directory used when loading files */
 private URL baseURL;

 /**\constructor"RLispJava(RClassLoader)"*/
 public RLispJava(RClassLoader rcl) {
  super();
  this.rcl = rcl;
  baseURL = null;
  try {
   baseURL = new URL("file:"+System.getProperty("user.dir")+File.separator);
  } catch(Throwable t) { System.err.println(t); }
 }

 /**\method"isSpecial(RPair)":
 Overrides parent method \See"RLisp#isSpecial(RPair)".
 */
 boolean isSpecial(RPair p) { return ( super.isSpecial(p) ||
   "string".equals(p.car()) ||
   "new".equals(p.car())    || "method".equals(p.car()) ||
   "field".equals(p.car())  || "array".equals(p.car())  ||
   "path".equals(p.car())   || "load".equals(p.car())   );
 }

 /**\method"evalSpecial(RPair, REnvironment)":
 Overrides parent method \See"RLisp#evalSpecial(RPair, REnvironment)".
 */
 Object evalSpecial(RPair p, REnvironment env) {
  if ( super.isSpecial(p) ) return( super.evalSpecial(p,env) );
  else return( evalJava(p,env) );
 }

 /**\method"evalJava(RPair, REnvironment)"*/
 Object evalJava(RPair je, REnvironment env) {
  //String op = je.nth(0).toString();    // operation
  Object op = eval(je.nth(0),env);
  if      ( "new".equals(op) )    return( evalJnew(je.CDR(),env) );
  else if ( "array".equals(op) )  return( evalJarray(je.CDR(),env) );
  else if ( "method".equals(op) ) return( evalJrun(je.CDR(),env) );
  else if ( "field".equals(op) )  return( evalJset(je.CDR(),env) );
  else if ( "path".equals(op) )   return( evalJpath(je.CDR(),env) );
  else if ( "load".equals(op) )   return( evalJload(je.CDR(),env) );
  else if ( "string".equals(op) ) return( je.CDR().toString(false) );
  else {
   System.err.println("ERROR: (" + op +  " ...) undefined");
   return(null);
  }
 }


 /**\method"evalJnew(RPair, REnvironment)"

 Note that "je" = "(Class arg0 arg1 ...)"

 We treat specially the case "argi" = "(cons 'Class object)".

 */
 Object evalJnew(RPair je, REnvironment env) {
  if (je == null || je.isNil() ) {
   System.err.println("ERROR: (new) found!");
   return(null);
  }
  Object result = null;
  try {
   Object co = eval(je.car(),env);
   if ( co == null ) return(null);
   Class cc = StoC(co.toString());
   RPair args = null; Object arg = null;
   int l = RPair.isList( je.CDR() );
   if ( l < 0 ) return(null); else args = (RPair)(je.CDR());
   Class[] argc = new Class[l];
   Object[] arga = new Object[l];
   for(int i=0; i<l; i++) {
    arg = args.nth(i);
    arga[i] = eval(arg,env);
    if ( arga[i] == null ) argc[i] = Void.TYPE;
    else {
     argc[i] = CtoC(arga[i].getClass());
     if ( RPair.isRPair(arga[i]) && ((RPair)arga[i]).car() != null ) {
      Class ca = StoC( ((RPair)arga[i]).car().toString() );
      if ( ca != null ) { argc[i] = ca;
       Object oa = ((RPair)arga[i]).cdr();
       if ( oa == null && ca.isInstance(RPair.nil) ) arga[i] = RPair.nil;
       else if ( oa == null || ca.isInstance(oa) ) arga[i] = oa;
       else arga[i] = StoO(ca,oa.toString());
      }
     }
    }
   }
   if ( cc.isArray() ) {
    Class cp = cc; while (cp.isArray()) cp = cp.getComponentType();
    int[] argi = new int[arga.length];
    for(int i=0; i<arga.length; i++)
     argi[i] = Integer.parseInt(arga[i].toString());
    result = Array.newInstance(cp, argi);
   } else result = cc.getConstructor(argc).newInstance(arga);
  } catch(Throwable te) {
   System.err.println("ERROR: (new "+je.toString(false)+") ["+te+"]");
   result = null;
  }
  return(result);
 }

 /**\method"evalJarray(RPair, REnvironment)"

 Note that "je" = "(Class ob0 ob1 ...)"
 */
 Object evalJarray(RPair je, REnvironment env) {
  Object result = null;
  if (je == null || je.isNil() ) {
   System.err.println("ERROR: (array) found!");
   return(null);
  }
  try{
   Object co = eval(je.car(),env);
   if ( co == null ) return(null);
   Class cc = StoC(co.toString());
   RPair args = null;
   int l = RPair.isList( je.CDR() );
   if ( l < 0 ) return(null); else args = (RPair)(je.CDR());
   result = Array.newInstance(cc,l);
   Object arg = null;
   for(int i=0; i<l; i++) {
    arg = eval(args.nth(i),env);
    if ( "String".getClass().equals(arg.getClass()) )
     arg = StoO(cc, (String)arg);
    Array.set(result, i, arg);
   }
  } catch(Throwable t) { System.err.println("ERROR: "+t); }
  return(result);
 }

 /**\method"evalJrun(RPair, REnvironment)"

 Note that "je" = "([Class | ob] Method arg0 arg1 ...))"

 We treat specially the case "argi" = "(cons 'Class object)".

 */
 Object evalJrun(RPair je, REnvironment env) {
  if (je == null || je.isNil() ) {
   System.err.println("ERROR: (method) found!");
   return(null);
  }
  Object result = null;
  Object o = eval(je.car(),env);
  Class cc = null;
  try { cc = Class.forName( o.toString(), true, rcl ); }
  catch(Throwable t) { cc = o.getClass(); } // o is not a Class name
  try {
   Object mo = eval(je.CDR().car(),env);
   if ( mo == null ) {
    System.err.println("ERROR: method not found!");
    return(null);
   }
   String mn = mo.toString();
   RPair args = null; Object arg = null;
   int l = RPair.isList( je.CDR().CDR() );
   if ( l < 0 ) return(null); else args = (RPair)(je.CDR().CDR());
   Class[] argc = new Class[l];
   Object[] arga = new Object[l];
   for(int i=0; i<l; i++) {
    arg = args.nth(i);
    arga[i] = eval(arg,env);
    if ( arga[i] == null ) argc[i] = Void.TYPE;
    else {
     argc[i] = CtoC(arga[i].getClass());
     if ( RPair.isRPair(arga[i]) && ((RPair)arga[i]).car() != null) {
      Class ca = StoC( ((RPair)arga[i]).car().toString() );
      if ( ca != null ) { argc[i] = ca;
       Object oa = ((RPair)arga[i]).cdr();
       if ( oa == null && ca.isInstance(RPair.nil) ) arga[i] = RPair.nil;
       else if ( oa == null || ca.isInstance(oa) ) arga[i] = oa;
       else arga[i] = StoO(ca,oa.toString());
      }
     }
    }
   }
   //Method met = cc.getDeclaredMethod(mn,argc);
   Method met = cc.getMethod(mn,argc);
   result = met.invoke(o,arga);
  } catch(Throwable te) {
   System.err.println("ERROR: (method "+je.toString(false)+") ["+te+"]");
   result = null;
  }
  return(result);
 }


 /**\method"evalJset(RPair, REnvironment)"

 Note that "je" = "([Class|ob] Field [val| ])"
 */
 Object evalJset(RPair je, REnvironment env) {
  if (je == null || je.isNil() ) {
   System.err.println("ERROR: (field) found!");
   return(null);
  }
  Object result = null;
  Object o = eval(je.car(),env);
  Class cc = null;
  try { cc = Class.forName( o.toString(), true, rcl ); }
  catch(Throwable t) { cc = o.getClass(); } // o is not a Class name
  try {
   Object fo = eval(je.CDR().car(),env);
   if ( fo == null ) {
    System.err.println("ERROR: field not found!");
    return(null);
   }
   String fn = fo.toString();
   //Field f = cc.getDeclaredField( fn );
   Field f = cc.getField( fn );
   if ( !RPair.isNil(je.CDR().CDR()) ) {
    Object v = eval(je.nth(2),env);
    if ( "String".getClass().equals( v.getClass() ) )
     v = StoO( f.getType(), (String)v );
    f.set(o, v);
   }
   result = f.get(o);
  }
  catch(Throwable te) {
   System.err.println("ERROR: (field "+je.toString(false)+") ["+te+"]");
   result = null;
  }
  return(result);
 }

 /**\method"evalJpath(RPair, REnvironment)"*/
 Object evalJpath(RPair je, REnvironment env) {
  String p = je.toString(false);
  try { rcl.addURL(new URL(baseURL, p)); }
  catch(Throwable t) { System.err.println(t); p = null; }
  return(p);
 }

 /**\method"evalJload(RPair, REnvironment)"

 A "(load URL)" calculates the location from a context.
 The base context is the user directory,
  {\catcode`\"=12\verb|System.getProperty("user.dir")|}.
 But each "(load URL)" sets the context to this "URL",
 so from file "Primes.lisp" to load a file "Maths.lisp" in the same
 directory just write "(load Maths.lisp)".

 For file addresses use:
  "(load file:path/filename.ext)".

 For files inside "jar" files use:\\
  "(load jar:file:path/file.jar!/inpath/filename.ext)".
 */
 Object evalJload(RPair je, REnvironment env) {
  Object res = null;
  URL oldbaseURL = baseURL;
  try {
   String urln = je.toString(false);
   URL url = new URL(baseURL, urln);
   // System.err.println("baseURL = "+baseURL);
   // System.err.println("    url = "+url);
   baseURL = url;
   InputStreamReader jisr = new InputStreamReader(url.openStream());
   BufferedReader in = new BufferedReader(jisr);
   String fc = "";
   String newline = in.readLine();
   while (newline != null) {
    fc = fc + newline + "\n";
    newline = in.readLine();
   }
   Object[] exp = RPair.Tokenize(fc);
   if (exp == null) return(null);
   for(int i=0; i<exp.length; i++) res = eval(exp[i],env);
  } catch(Throwable t) {
   System.err.println(t);
   res = null;
  }
  baseURL = oldbaseURL;
  return(res);
 }


 /**\method"CtoC(Class)"

 If the "Class" "oc" is a primitive type enclosing class, then returns
 the "Class" object representing the primitive type.
 Otherwise it returns "oc".
 */
 private Class CtoC(Class oc) {
  Class c = oc;
  if (oc==null) return(null);
  else if (oc.equals(Boolean.class)) c = Boolean.TYPE;
  else if (oc.equals(Character.class)) c = Character.TYPE;
  else if (oc.equals(Integer.class)) c = Integer.TYPE;
  else if (oc.equals(Byte.class)) c = Byte.TYPE;
  else if (oc.equals(Short.class)) c = Short.TYPE;
  else if (oc.equals(Long.class)) c = Long.TYPE;
  else if (oc.equals(Float.class)) c = Float.TYPE;
  else if (oc.equals(Double.class)) c = Double.TYPE;
  return(c);
 }

 /** \method"StoC(String)"

 Given a name, it returns the Class using the incremental Class loader.
 A bidimensional array of base class Class is noted Class[][].

 @param cn is the Class name
 @return the Class object
 */
 private Class StoC(String cn) {
  if ( cn == null ) return(null);
  Class c;
  int dims = 0; int l = cn.length();
  while ( cn.lastIndexOf("[]") == l-2 ) { dims++;
   cn = cn.substring(0,l-2); l = cn.length();
  }
  if (cn.equals("java.lang.String")) c = "String".getClass();
  else if (cn.equals("String")) c = "String".getClass();
  else if (cn.equals("boolean")) c = Boolean.TYPE;
  else if (cn.equals("char")) c = Character.TYPE;
  else if (cn.equals("int")) c = Integer.TYPE;
  else if (cn.equals("byte")) c = Byte.TYPE;
  else if (cn.equals("short")) c = Short.TYPE;
  else if (cn.equals("long")) c = Long.TYPE;
  else if (cn.equals("float")) c = Float.TYPE;
  else if (cn.equals("double")) c = Double.TYPE;
  else if (cn.equals("void")) c = Void.TYPE;
  else try { c = Class.forName(cn,true,rcl); }
  catch(ClassNotFoundException e) {
   System.err.println("Class not found: " + cn);
   c = null;
  }
  if (c != null && dims > 0) c = arrayClass(c,dims);
  return(c);
 }

 /** \method"arrayClass(Class, int)"

 Given a base type and a number of dimensions,
 it returns the corresponding array class.

 */
 public static Class arrayClass(Class c, int dims) {
  if ( c == null || dims < 0 ) return(null);
  if ( dims == 0 ) return(c);
  int[] d = new int[dims]; for(int i=0; i<dims; i++) d[i] = 0;
  Class ac = null;
  try{ ac = Array.newInstance(c,d).getClass(); }
  catch(Throwable t) { System.err.println("ERROR: " + t); }
  return(ac);
 }

 /** \method"StoO(Class, String)"

 Given a Class and the name of one value,
 it returns the corresponding object.

 */
 public static Object StoO(Class c, String on) {
   if ( c == null || on == null ) return(null);
   Object o = null;
   if ( c.isInstance(on) ) o = on;
   else if ( c.equals("String".getClass()) ) o = on;
   else if (c.equals(Boolean.TYPE)) o = new Boolean(on);
   else if (c.equals(Character.TYPE)) o = new Character(on.charAt(0));
   else if (c.equals(Integer.TYPE)) o = new Integer(on);
   else if (c.equals(Byte.TYPE)) o = new Byte(on);
   else if (c.equals(Short.TYPE)) o = new Short(on);
   else if (c.equals(Long.TYPE)) o = new Long(on);
   else if (c.equals(Float.TYPE)) o = new Float(on);
   else if (c.equals(Double.TYPE)) o = new Double(on);
   else if (c.equals(Void.TYPE)) o = null;
   else {
    String[] arg = new String[1]; arg[0] = on;
    Class[] carg = new Class[1]; carg[0] = "String".getClass();
    //try { o = c.getDeclaredConstructor(carg).newInstance(arg); }
    try { o = c.getConstructor(carg).newInstance(arg); }
    catch(Throwable t) { System.err.println(t); o = null; }
   }
   return(o);
  }

 /**\method"toString()"*/
 public String toString() { return("RLispJava"); }

}
