/** \class "RLispConsole"

An "RLispConsole" is a Graphical User Interface (GUI) that implements a
Java interpreter. This interpreter uses a complete Lisp interpreter,
with its own environment to store the named objects, so the Java
interpreter syntax is Lispian (or Schemian).

It allows:
 "new") to define a Java object,
 "array") or a matrix,
 "path") from a directory (also known as folder),
 "name") and give it a name,
 "unname") or took it away;
 "run") to run a method,
 "set") and to see or set a field value.

@author  Ramn Casares 2003
@version 2003.03.19
*/
package RLisp;

import java.net.URL;

import java.lang.reflect.Constructor;
import java.lang.reflect.Method;
import java.lang.reflect.Field;
import java.lang.reflect.Array;
import java.lang.reflect.Modifier;
import java.lang.reflect.InvocationTargetException;

import java.util.Vector;
import java.util.BitSet;
import java.util.Date;

import java.io.PrintWriter;
import java.io.BufferedWriter;
import java.io.FileWriter;
import java.io.PrintStream;
import java.io.BufferedReader;
import java.io.InputStreamReader;
import java.io.InputStream;
import java.io.FileReader;

import java.io.IOException;
import java.io.FileNotFoundException;
import javax.swing.text.BadLocationException;

import java.util.*;
import java.awt.*;
import java.io.File;
import javax.swing.*;
import java.awt.event.*;
import java.io.PipedWriter;
import java.io.PipedReader;
import java.io.BufferedReader;
import java.io.IOException;

import javax.swing.JOptionPane;

public class RLispConsole implements WindowListener, ActionListener {

 public RLispInterpreter lisp;

 private Object result; // referred to as @
 private String expression;
 private String logfilename = "RLisp.log"; // default name
 private boolean logging = false;
 private PrintWriter outfile = null;
 private String version = "20040115";

 /**\variable"rcl"

 It is an incremental "ClassLoader" that is used for loading all of the
 classes. For the Java Virtual Machine, the same ".class" file
 loaded twice from the same directory by two different "ClassLoader"s,
 are two completelly different classes. Because of this, the same
 "ClassLoader" should load every class.
 */
 private RClassLoader rcl;

 JFrame frame;
 JTextArea textArea;
 JLabel statuslabel;
 Container contentPane;

 /** \constructor "RLispConsole(String)" */
 public RLispConsole(String title) {

  URL[] urls = new URL[1];
  File ud = new File(System.getProperty("user.dir"));
  try { urls[0] = ud.toURL(); }
  catch (java.net.MalformedURLException mue) {} // always right
  rcl = new RClassLoader(urls);

  lisp = new RLispInterpreter( new RLispJava(rcl) );

  frame = new JFrame(title);

  frame.addNotify();
  frame.addWindowListener(this);

  frame.setDefaultCloseOperation(WindowConstants.DISPOSE_ON_CLOSE);

  //setBounds(200, 200, 500, 200);

  contentPane = frame.getContentPane();
  //contentPane.setLayout(new FlowLayout());
  //contentPane.setLayout(new GridLayout(3,1));
  //contentPane.setLayout(new BoxLayout(contentPane,BoxLayout.PAGE_AXIS));
  contentPane.setLayout(new BoxLayout(contentPane,BoxLayout.Y_AXIS));

  JMenuBar Bar = new JMenuBar();
   JMenu menuFile = new JMenu("File"); ////////////////////
    JMenuItem miLoad = new JMenuItem("Load");
     miLoad.addActionListener(this);
     menuFile.add(miLoad);
    JMenuItem miLoadFrom = new JMenuItem("Load from");
     miLoadFrom.addActionListener(this);
     menuFile.add(miLoadFrom);
    JMenuItem miSave = new JMenuItem("Save");
     miSave.addActionListener(this);
     menuFile.add(miSave);
    JMenuItem miSaveTo = new JMenuItem("Save to");
     miSaveTo.addActionListener(this);
     menuFile.add(miSaveTo);
     menuFile.addSeparator();
    JMenuItem miTree = new JMenuItem("Tree");
     miTree.addActionListener(this);
     menuFile.add(miTree);
    JMenuItem miKeyboard = new JMenuItem("Keyboard");
     miKeyboard.addActionListener(this);
     menuFile.add(miKeyboard);
    JMenuItem miSession = new JMenuItem("Session");
     miSession.addActionListener(this);
     menuFile.add(miSession);
    menuFile.addSeparator();
    JMenuItem miClose = new JMenuItem("Close");
     miClose.addActionListener(this);
     menuFile.add(miClose);
   JMenu menuEdit = new JMenu("Edit"); ////////////////////
    JMenuItem miCut = new JMenuItem("Cut");
     miCut.setEnabled(false);
     miCut.addActionListener(this);
     menuEdit.add(miCut);
    JMenuItem miCopy = new JMenuItem("Copy");
     miCopy.addActionListener(this);
     menuEdit.add(miCopy);
    JMenuItem miPaste = new JMenuItem("Paste");
     miPaste.setEnabled(false);
     miPaste.addActionListener(this);
     menuEdit.add(miPaste);
   JMenu menuAction = new JMenu("Action"); ////////////////
    JMenuItem miPath = new JMenuItem("Path");
     miPath.addActionListener(this);
     menuAction.add(miPath);
    JMenuItem miLisp = new JMenuItem("Lisp code");
     miLisp.addActionListener(this);
     menuAction.add(miLisp);
    JMenuItem miName = new JMenuItem("Name");
     miName.addActionListener(this);
     menuAction.add(miName);
    JMenuItem miUnname = new JMenuItem("Unname");
     miUnname.addActionListener(this);
     menuAction.add(miUnname);
    JMenuItem miList = new JMenuItem("List");
     miList.addActionListener(this);
     menuAction.add(miList);
   JMenu menuHelp = new JMenu("Help"); ////////////////////
    JMenuItem miManual = new JMenuItem("Manual");
     miManual.addActionListener(this);
     menuHelp.add(miManual);
    JMenuItem smiManual = new JMenuItem("Spanish Manual");
     smiManual.addActionListener(this);
     menuHelp.add(smiManual);
    JMenuItem miCode = new JMenuItem("Code");
     miCode.addActionListener(this);
     menuHelp.add(miCode);
    JMenuItem miAbout = new JMenuItem("About RLisp");
     miAbout.addActionListener(this);
     menuHelp.add(miAbout);

  Bar.add(menuFile);
  Bar.add(menuEdit);
  Bar.add(menuAction);
  Bar.add(menuHelp);
  frame.setJMenuBar(Bar);

  textArea = new JTextArea(15,50);
  textArea.setEditable(false);
  textArea.setLineWrap(false);
  textArea.setBackground(new Color(1.0F,1.0F,0.5F)); // yellow
  textArea.setFont(new Font("Monospaced",Font.PLAIN,12));

  contentPane.add(new JScrollPane(textArea,
   ScrollPaneConstants.VERTICAL_SCROLLBAR_AS_NEEDED,
   ScrollPaneConstants.HORIZONTAL_SCROLLBAR_AS_NEEDED));

  Box statusbox = Box.createHorizontalBox();
   statuslabel = new JLabel("Status: ");
   statusbox.add(statuslabel);
   statusbox.add(Box.createHorizontalGlue());
   contentPane.add(statusbox);

  frame.setVisible(true);
  frame.pack();
  frame.show();
 }

 /** \method"initLisp()"

 If the "RLisp" is running from "RLisp.jar", then it loads
 the file \See"file#RLisp.lisp" in the "jar" file.
 Otherwise it loads the file \See"file#RLisp.lisp" which is
 in the same directory that the "RLispConsole.class" that is running.
 */
 public void initLisp() {
  try{
   URL lispURL =
    this.getClass().getClassLoader().getResource("RLisp/RLisp.lisp");
   String sURL = lispURL.toString();
   writeln("<< (load " + sURL + ")");
   lisp.eval("(load " + sURL + ")");
   System.out.println("Init Lisp [" + lisp.counter(0) + "]");
  } catch(Throwable t) { System.err.println(t); }
 }

 /** \method"setLogFile(String)" */
 public void setLogFile(String filename) { this.logfilename = filename; }

 /** \method"readFile(String)" */
 public void readFile(String filename) throws IOException {
  BufferedReader infile = new BufferedReader(new FileReader(filename));
  String line = infile.readLine();
  while ( (line != null) && !("<< quit".equals(line)) ) {
   textArea.append(line + "\n");
   if ( line.startsWith("<< ") ) {
    expression = line.substring(3);
    result = lisp.eval(expression);
    textArea.append(">< " + resultToString() + "\n" );
    lisp.ENV.define("@",result);
   }
   line = infile.readLine();
  }
  infile.close();
  if ("<< quit".equals(line)) closeAction();
 }

 /** \method"session()"

 It runs a session in the system console.
 If there were not a system console, because Java was call as "javaw",
 then control would be lost and should be recovered manually
 by pressing Ctr-Alt-Del and then aborting task "javaw".
 */
 public void session() throws IOException {
  frame.hide();
  BufferedReader in = new BufferedReader(new InputStreamReader(System.in));
  String outputLine = "New session: " + new Date();
  System.out.println(">> "+ outputLine); writeln(">> " + outputLine);
  System.out.println("<>  To end the session, enter \"quit\"");
  String oldLine = "quit";
  System.out.print("<< "); expression = in.readLine();
  if (expression.equals("")) expression = oldLine;
  while ( !("quit".equals(expression)) ) {
   writeln("<< " + expression);
   result = lisp.eval(expression);
   outputLine = resultToString();
   System.out.println(">> " + outputLine);
   writeln(">> " + outputLine);
   lisp.ENV.define("@",result);
   oldLine = expression;
   System.out.print("<< "); expression = in.readLine();
   if (expression.equals("")) expression = oldLine;
  }
  writeln("<> Session finished: " + new Date());
  frame.show();
 }

 /** \method"write(String)" */
 public void write(String s) {
  textArea.append(s);
  if ( outfile != null ) outfile.print(s);
 }

 /** \method"writeln(String)" */
 public void writeln(String s) {
  textArea.append(s); textArea.append("\n");
  if ( logging ) outfile.println(s);
 }

 /** \method"writeln()" */
 public void writeln() { writeln(""); }


 /** \method"executeObject(Object)" */
 public void executeObject(Object rro) {
  if ( rro == null ) return;
  Object[] values = null;
  Object value = null;
  String keyselected = null;

 try {

  if ( rro.getClass() == Class.forName("java.lang.reflect.Constructor") ) {
   Constructor c = (Constructor)rro;
   expression = "(new \'" + c.getName();
   values = getValues( c.getParameterTypes() );
   expression = expression + ")";
   writeln("<< " + expression);
   if ( values != null ) {
    result = c.newInstance(values);
    writeln(">> " + resultToString());
    lisp.ENV.define("@",result);
   } else { System.err.println("new " + rro + " ERROR!"); }

 } else if ( rro.getClass() == Class.forName("java.lang.reflect.Method") ) {
   Method c = (Method)rro;
   Class cl = c.getDeclaringClass();
   Object obj = null;
   if ( Modifier.isStatic(c.getModifiers()) ) {
    expression = "(method \'" + cl.getName() + " \'" + c.getName();
    values = getValues( c.getParameterTypes() );
    expression = expression + ")";
    writeln("<< " + expression);
   } else {
    Object[] objs = lisp.ENV.keys(cl,false);
    if (objs != null && objs.length > 0 ) {
     keyselected = (String)JOptionPane.showInputDialog(null,
                    getName(cl), "Choose an object", // toString
                    JOptionPane.QUESTION_MESSAGE, null,
                    objs, objs[0] );
     if ( keyselected == null ) values = null;
     else {
      obj = lisp.ENV.lookup( keyselected );
      expression = "(method " + keyselected + " \'" + c.getName();
      values = getValues( c.getParameterTypes() );
      expression = expression + ")";
      writeln("<< " + expression);
     }
    }
   }
   if ( values != null ) {
    result = c.invoke(obj,values);
    if ( result == null ) writeln(">> [null]");
    else writeln(">> " + resultToString());
    lisp.ENV.define("@",result);
   } else { System.err.println("run " + rro + " ERROR!"); }

 } else if ( rro.getClass().isArray() ) {
  Class c = rro.getClass().getComponentType();
  expression = "(array \'" + c.getName();
  Vector vals = new Vector();
  Object val = getValue(c);
  while ( val != null ) { vals.add(val); val = getValue(c); }
  expression = expression + ")";
  writeln("<< " + expression);
  Object[] valsa = vals.toArray();
  int l = valsa.length;
  result = Array.newInstance(c,l);
  for(int i=0; i<l; i++) Array.set(result, i, valsa[i]);
  writeln(">> " + resultToString());
  lisp.ENV.define("@",result);

 } else if ( rro.getClass() == Class.forName("java.lang.reflect.Field") ) {
  Field f = (Field)rro;
  Class cl = f.getDeclaringClass();
  Object obj = null;
  String owner = null;
  if ( Modifier.isStatic( f.getModifiers() ) ) {
   owner = "\'" + cl.getName();
  } else {
   Object[] objs = lisp.ENV.keys(cl,false);
   if (objs != null && objs.length > 0 ) {
    keyselected = (String)JOptionPane.showInputDialog(null,
                   getName(cl), "Choose an object", // toString
                   JOptionPane.QUESTION_MESSAGE, null,
                   objs, objs[0] );
    if ( keyselected != null ) {
     obj = lisp.ENV.lookup( keyselected );
     owner = keyselected;
    }
   }
  }
  Object currentValue = f.get(obj);
  Class fc = f.getType();
  if ( Modifier.isFinal( f.getModifiers() ) ) {
   JOptionPane.showMessageDialog(null,
    "Final value: " + prettyPrint(currentValue),
    "Field " + f.toString(),
    JOptionPane.INFORMATION_MESSAGE);
   expression = "(field " + owner + " \'" + f.getName() + ")";
   writeln("<< " + expression);
   result = currentValue;
   writeln(">> " + resultToString());
   lisp.ENV.define("@",result);
  } else if (
   JOptionPane.showConfirmDialog(null,
     "Current value: " + prettyPrint(currentValue) + "\n" + // toString
     "Do you want to change it?",
     "Field " + f.toString(),
     JOptionPane.YES_NO_OPTION,
     JOptionPane.INFORMATION_MESSAGE) == JOptionPane.YES_OPTION ) {
    expression = "(field " + owner + " \'" + f.getName();
    currentValue = getValue(fc);
    f.set(obj, currentValue);
    expression = expression + ")";
    writeln("<< " + expression);
    result = f.get(obj);
    writeln(">> " + resultToString());
    lisp.ENV.define("@",result);
   } else { // NO OPTION
    expression = "(field " + owner + " \'" + f.getName() + ")";
    writeln("<< " + expression);
    result = currentValue;
    writeln(">> " + resultToString());
    lisp.ENV.define("@",result);
   }
  } else System.err.println("ERROR: No action for class "+rro.getClass());
 }
 catch (ClassNotFoundException cnfe) { System.err.println(cnfe); }
 catch (IllegalAccessException iae) { System.err.println(iae); }
 catch (InvocationTargetException ite) { System.err.println(ite); }
 catch (InstantiationException ie) { System.err.println(ie); }
 catch (IllegalArgumentException iae) { System.err.println(iae); }
 }


 /** \method "getValue(Class)"

 It gets from the user an object of the given "Class".

 @param c is the given "Class"
 @return the chosen object
 */
 Object getValue(Class c) {
  if ( c == null ) return(null);
  Object value = null;
  String keyselected = null;
  String sin = getName(c);
  String sout = null;
  Object[] names = lisp.ENV.keys(c,false);
  if ( names == null || names.length == 0 ) {
   sout = JOptionPane.showInputDialog(sin + " expression");
   if ( sout == null ) return(null);
   value = lisp.eval(sout);
   expression = expression + " " + sout;
  } else {
   Object[] namess = new Object[names.length+1];
   namess[0] = sin + " expression";
   for(int j=0; j<names.length; j++) namess[j+1] = names[j];
   keyselected = (String)JOptionPane.showInputDialog(null,
                    sin, "Input value",
                    JOptionPane.QUESTION_MESSAGE, null,
                    namess, namess[0] );
   if ( keyselected == null ) return(null);
   else if ( keyselected.equals(sin + " expression") ) {
    sout = JOptionPane.showInputDialog(sin + " expression");
    if ( sout == null ) return(null);
    value = lisp.eval(sout);
    expression = expression + " " + sout;
   } else {
    value = lisp.ENV.lookup( keyselected );
    expression = expression + " " + keyselected;
   }
  }
  if ( value == null ) return(null);
  if ( "String".getClass().equals(value.getClass()) )
   value = RLispJava.StoO(c,(String)value);
  return(value);
 }

 /** \method"getValues(Class[])"

 It gets from the user an "array" of "Object"s of the given "Class"es.

 @param c is the "array" of given "Class"es
 @return the "array" of chosen "Object"s
 */
 Object[] getValues(Class[] ca) {
  if ( ca == null ) return(null);
  Object[] values = new Object[ca.length];
  String keyselected; Object value;
  String sin; String sout;
  for(int i=0; i<ca.length; i++) {
   keyselected = null; value = null;
   sin = getName(ca[i]); sout = null;
   Object[] names = lisp.ENV.keys(ca[i],false);
   if ( names == null || names.length == 0 ) {
    sout = JOptionPane.showInputDialog(sin + " expression");
    if ( sout == null ) return(null);
    value = lisp.eval(sout);
    expression = expression + " " +
                      "(cons \'" + sin + " " + sout + ")";
   } else {
    Object[] namess = new Object[names.length+1];
    namess[0] = sin + " expression";
    for(int j=0; j<names.length; j++) namess[j+1] = names[j];
    keyselected = (String)JOptionPane.showInputDialog(null,
                     sin, "Input value["+i+"]",
                     JOptionPane.QUESTION_MESSAGE, null,
                     namess, namess[0] );
    if ( keyselected == null ) return(null);
    else if ( keyselected.equals(sin + " expression") ) {
     sout = JOptionPane.showInputDialog(sin + " expression");
     if ( sout == null ) return(null);
     value = lisp.eval(sout);
     expression = expression + " " +
                       "(cons \'" + sin + " " + sout + ")";
    } else {
     value = lisp.ENV.lookup( keyselected );
     expression = expression + " " +
                  "(cons \'" + sin + " " + keyselected + ")";
    }
   }
   if ( value == null ) return(null);
   else {
    if ( "String".getClass().equals(value.getClass()) )
     values[i] = RLispJava.StoO(ca[i],(String)value);
    else
     values[i] = value;
   }
  }
  return values;
 }

 /** \method"getName(Class)"

 Works as Class.getName(), except with arrays.

 @param c is the Class
 @return its name
 */
 public static String getName(Class c) {
  if ( c == null ) return(null);
  String cis = c.getName();
  if ( c.isArray() ) {
   String atend = "";
   Class pc = c;
   while ( cis.charAt(0) == '[' ) {
    atend = atend + "[]";
    pc = pc.getComponentType();
    cis = cis.substring(1);
   }
   cis = pc.getName() + atend;
  }
  return(cis);
 }

 /**\method"windowClosing(WindowEvent)"

 Implements interface "WindowListener".
 The only no void method is "windowClosing(WindowEvent)"
 */
 public void windowOpened(WindowEvent e) {}
 public void windowClosing(WindowEvent e) { closeAction(); }
 public void windowClosed(WindowEvent e) {}
 public void windowIconified(WindowEvent e) {}
 public void windowDeiconified(WindowEvent e) {}
 public void windowActivated(WindowEvent e) {}
 public void windowDeactivated(WindowEvent e) {}


 /**\method"actionPerformed(ActionEvent)"

 Implements the "ActionListener" interface.
 */
 public void actionPerformed(ActionEvent e) {
  String texto = e.getActionCommand();
  statuslabel.setText(texto);
  if      ("Cut".equals(texto))       textArea.cut();
  else if ("Copy".equals(texto))      textArea.copy();
  else if ("Paste".equals(texto))     textArea.paste();
  else if ("Load".equals(texto))      loadAction();
  else if ("Load from".equals(texto)) loadFromAction();
  else if ("Save".equals(texto))      saveAction();
  else if ("Save to".equals(texto))   saveToAction();
  else if ("Close".equals(texto))     closeAction();
  else if ("Tree".equals(texto))      treeAction();
  else if ("Name".equals(texto))      nameAction();
  else if ("Unname".equals(texto))    unnameAction();
  else if ("List".equals(texto))      listAction();
  else if ("Path".equals(texto))      pathAction();
  else if ("Lisp code".equals(texto)) loadLispAction();
  else if ("Keyboard".equals(texto))  keyboardAction();
  else if ("Line".equals(texto))      lineAction(e.getSource());
  else if ("Session".equals(texto))   sessionAction();
  else if ("OK".equals(texto))        okTreeAction(e.getSource());
  else if ("Manual".equals(texto))    pdfAction("RLispManE.pdf");
  else if ("Spanish Manual".equals(texto)) pdfAction("RLispManS.pdf");
  else if ("Code".equals(texto))      pdfAction("RLispCode.pdf");
  else if ("About RLisp".equals(texto)) aboutAction();
  else System.err.println("ERROR: Action " + texto + " undefined!");
  System.out.println(texto + " [" + lisp.counter(0) + "]");
 }

 /** \method"closeAction()" */
 private void closeAction() {
  writeln("<> Closing: " + new Date());
  if (outfile != null) outfile.close();
  System.exit(0);
 }

 /** \method"lineAction(Object)" */
 private void lineAction(Object so) {
  RButton rb = (RButton)so;
  expression = (String)rb.getObject();
  writeln("<< " + expression);
  if ( "quit".equals(expression) ) closeAction();
  else {
   result = lisp.eval(expression);
   writeln(">> " + resultToString());
   lisp.ENV.define("@",result);
  }
 }

 /** \method"okTreeAction(Object)" */
 private void okTreeAction(Object so) {
  if (so == null) return;
  RButton rb = (RButton)so;
  RClassTree rct = (RClassTree)rb.getObject();
  executeObject(rct.getSelectedObject());
 }

 /** \method"sessionAction()" */
 private void sessionAction() {
  try { session(); } catch (Throwable t) {System.err.println(t);}
 }

 /** \method"keyboardAction()" */
 private void keyboardAction() {
  new RKeyboard(this,"(quote Keyboard)");
 }

 /** \method"listAction()" */
 private void listAction() {
  Object[] names = lisp.ENV.keys();
  Object value; String vn;
  writeln("<< names in " + lisp.ENV);
  for (int i=0; i<names.length; i++) {
   value = lisp.ENV.lookup(names[i]);
   writeln( ">> >> " + getName(value.getClass()) + " " +
   names[i] + " = " + prettyPrint(value));
  }
 }

 /** \method"unnameAction()" */
 private void unnameAction() {
  Object[] names = lisp.ENV.keys();
  if (names != null && names.length > 0) {
   Object keyselected = (String)JOptionPane.showInputDialog(null,
                     "Unname", "Select key to delete",
                     JOptionPane.QUESTION_MESSAGE, null,
                     names, names[0] );
   if ( keyselected != null ) {
    expression = "(set! " + keyselected + ")";
    writeln("<< " + expression);
    result = lisp.ENV.set( keyselected, null );
    writeln(">> " + resultToString());
    lisp.ENV.define("@",result);
   }
  }
 }

 /** \method"nameAction()" */
 private void nameAction() {
  if ( result == null) {
   JOptionPane.showMessageDialog(null,"null can not be named!");
  } else {
   String title = cutString("Name for " + resultToString(),32);
   String name = JOptionPane.showInputDialog(title);
   if ( name != null) {
    expression = "(def " + name + " @)";
    writeln("<< " + expression);
    result = lisp.ENV.define(name,result);
    writeln(">> " + resultToString());
    lisp.ENV.define("@",result);
   }
  }
 }

 /** \method"pathAction()" */
 private void pathAction() {
  JFileChooser chooser = new JFileChooser(System.getProperty("user.dir"));
  chooser.setFileSelectionMode(JFileChooser.DIRECTORIES_ONLY);
  if(chooser.showOpenDialog(null) == JFileChooser.APPROVE_OPTION) {
   File cd = chooser.getSelectedFile();
   try {
    rcl.addURL( cd.toURI().toURL() );
    writeln("<< (path " + cd.toURI().toURL() + ")" );
   } catch(java.net.MalformedURLException mue) { System.err.println(mue); }
  }
 }

 /** \method"treeAction()" */
 private void treeAction() {
  try {
   RClassTree rct = new RClassTree(this,true,rcl);
   writeln("<> Tree from  " + rct.cd.toURI().toURL() );
   writeln("<< (path " + rct.cd.toURI().toURL() + ")");
  } catch(Throwable t) { System.err.println(t); }
 }

 /** \method"loadAction()" */
 private void loadAction() {
  writeln("<> Loading from " + logfilename);
  try { readFile(logfilename); } catch (Throwable t) {System.err.println(t);}
  writeln("<> Loaded " + logfilename);
 }

 /** \method"loadFromAction()" */
 private void loadFromAction() {
  JFileChooser chooser = new JFileChooser(System.getProperty("user.dir"));
  //chooser.setFileSelectionMode(JFileChooser.FILES_ONLY);
  chooser.setFileFilter(new RExtFilter(".log"));
  if(chooser.showOpenDialog(null) == JFileChooser.APPROVE_OPTION) {
   File cd = chooser.getSelectedFile();
   if ( cd.canRead() ) {
    try {
     String filename = cd.getAbsolutePath();
     writeln("<> Loading from " + filename);
     rcl.addURL( cd.getParentFile().toURL() );
     readFile(filename);
     writeln("<> Loaded " + filename);
    } catch (Throwable t) {System.err.println(t);}
   }
  }
 }

 /** \method"loadLispAction()" */
 private void loadLispAction() {
  JFileChooser chooser = new JFileChooser(System.getProperty("user.dir"));
  //chooser.setFileSelectionMode(JFileChooser.FILES_ONLY);
  chooser.setFileFilter(new RExtFilter(".lisp"));
  if(chooser.showOpenDialog(null) == JFileChooser.APPROVE_OPTION) {
   File cd = chooser.getSelectedFile();
   if ( cd.canRead() ) {
    try {
     URL filename = cd.toURI().toURL();
     URL path = cd.getParentFile().toURI().toURL();
     writeln("<< (path " + path.toString() + ")");
     rcl.addURL(path);
     writeln("<< (load " + filename + ")");
     lisp.eval("(load " + filename + ")");
    } catch (Throwable t) {System.err.println(t);}
   }
  }
 }

 /** \method"saveAction()" */
 private void saveAction() {
  try {
   writeln("<> Saving to " + logfilename);
   logging = true;
   outfile = new PrintWriter(new BufferedWriter
    (new FileWriter(logfilename,true)));
   writeln("<> Date: " + new Date());
  } catch (Throwable t) {System.err.println(t);} //t.printStackTrace();}
 }

 /** \method"saveToAction()" */
 private void saveToAction() {
  JFileChooser chooser = new JFileChooser(System.getProperty("user.dir"));
  if(chooser.showSaveDialog(null) == JFileChooser.APPROVE_OPTION) {
   File cd = chooser.getSelectedFile();
   String filename = cd.getAbsolutePath();
   if ( filename != null && filename.length() >0 ) {
    logfilename = filename;
    saveAction();
   }
  }
 }

 /** \method"pdfAction(String)"

 The user manual is "RLispMan.pdf".
 The code is in "RLispCode.pdf".
 */
 private void pdfAction(String fn) {
  URL ju = this.getClass().getClassLoader().getResource("RLisp.jar");
  if (ju==null)
   ju = this.getClass().getClassLoader().getResource("RLisp/RLispConsole.class");
  try{
   URL fu = new URL(ju,fn);
   String name = fu.getFile().substring(1);
   // name = name.replaceAll("%20"," "); // is Java 1.4
    int i = name.indexOf("%20");
    while (i >= 0) {
     name = (new StringBuffer(name)).replace(i,i+3," ").toString();
     i = name.indexOf("%20");
    }
   Runtime.getRuntime().exec("Start \"/MAX\" \"" + name + "\""); // Windows
   writeln("<> Start \"/MAX\" \"" + name + "\"");
  } catch (Throwable t) {
   writeln("<> File " + fn + " not found!");
   System.err.println(t);
  }
 }

 /** \method"aboutAction()" */
 private void aboutAction() {
  try {
   String[] message = new String[3];
   message[0] = "RLisp " + version;
   message[1] = " 2004 Ramn Casares";
   message[2] = "r.casares@ieee.org";
   JOptionPane.showMessageDialog(null, message, "About RLisp",
    JOptionPane.INFORMATION_MESSAGE);
   writeln("<> RLisp "+version+ " ("+lisp+")");
  } catch (Throwable t) {System.err.println(t);}
 }

 /** \method "toArray(String)" */
 public static String[] toArray(String sentence) {
  StringTokenizer st = new StringTokenizer(sentence," ");
  int l = st.countTokens();
  String[] word = new String[l];
  for(int i=0; i<l; i++) word[i] = st.nextToken();
  return(word);
 }

 /** \method "cutString(String, int)" */
 public static String cutString(String s, int i) {
  if( s.length() > i ) return(s.substring(0,i)+"...");
  else return(s);
 }

 /** \method "arrayToString(Object[])" */
 public static String arrayToString(Object[] array) {
  String s = "{";
  for(int i=0; i<array.length; i++) {
   if (i>0) s = s + ", ";
   if ( array[i] == null ) s = s + "[null]";
   else s = s + prettyPrint(array[i]);
  }
  s = s + "}";
  return(s);
 }

 /** \method "prettyPrint(Object)" */
 public static String prettyPrint(Object o){
  String s = "ERROR!";
  if ( o == null ) s = "[null]";
  else if ( o.getClass().isArray() ) {
   Object[] oa = new Object[Array.getLength(o)];
   for(int i=0; i<oa.length; i++) oa[i] = Array.get(o,i);
   s = arrayToString(oa);
  } else s = o.toString();
  return(s);
 }

 /** \method "resultToString()" */
 private String resultToString() { return(prettyPrint(result)); }

 /** \method "main(String[])"

 It creates a console to play the Java "Object"s accessible
 from the curren directory.

 @param args are the command line arguments
 */
 public static void main(String[] args) {
  RLispConsole rlc = new RLispConsole("RLisp");
  rlc.initLisp();
  if ( args.length > 0 ) {
   String s = "";
   for(int i=0; i<args.length; i++) s = s + " " + args[i];
   s = s.substring(1);
   try {
    rlc.writeln("<> Loading from " + s);
    rlc.readFile(s);
    rlc.writeln("<> Loaded " + s);
    System.out.println("Load from [" + rlc.lisp.counter(0) + "]");
   } catch(IOException ioe) {
    System.err.println("ERROR: file " + s + " not found!");
   }
  }
 }

}
