/** \class "RLisp"

 Implements a minimum Lisp evaluator.

@author  Ramn Casares 2003
@version 2003.02.05
*/
package RLisp;

public class RLisp {

 /**\variable"t" is the "true" constant */
 public static final Boolean t = new Boolean("true");
 /** \method"isTrue(Object)" */
 public static boolean isTrue(Object o) {
  return( o != null && o instanceof Boolean &&
          ((Boolean)o).booleanValue() );
 }

 /**\variable"counter" counts eval cycles */
 public int counter = 0;

 /**\method"eval(Object, REnvironment)"

 The main evaluator.

 It evaluates:
 {\medskipamount=0pt
 \point "key" to its value as defined in "env".
 \point any other no \See"RPair" object to itself.
 \point "(quote anything)" $\rightarrow$ "anything".
 \point "(eval expression [env])" evaluates "expression" and then
        evaluates the result of the evaluation.
 \point "(def key value)" binds "key" to "value" in "env".
 \point "(set! key value)" changes the binding of "key" to "value".
 \point "(cond (b0 e01 e02 ...) (b1 e11 e12 ...) ... )" $\rightarrow$
        if "b0" to "e10 e02 ...", else if "b1" to "e11 e12 ...", \dots.
 \point "(lambda (f0 f1 ... ) e0 e1 e2 ... )" $\rightarrow$ the function.
 \point "(rho name expander)" $\rightarrow$ the special form.
 \point "(function a0 a1 ... )" applies the "function" using arguments
        "a0" "a1" \dots.
 }
 */
 public Object eval(Object exp, REnvironment env) { counter++;
  if ( exp == null ) return(null);
  if ( RPair.isAtom(exp) ) {
   Object v = env.lookup(exp);
   if ( v == null ) return(exp);  // self-evaluating
   else             return(v);    // identifier
  } else { RPair re = (RPair)exp;
   if ( isSpecial(re) ) return( evalSpecial(re,env) ); // hook to extend
   else if ( "quote".equals(re.car()) )  return( re.CDR().car() );
   else if ( "eval".equals(re.car()) )   return( evalEval(re,env) );
   else if ( "set!".equals(re.car()) )   return( evalSet(re,env) );
   else if ( "def".equals(re.car()) )    return( evalDef(re,env) );
   else if ( "cond".equals(re.car()) )   return( evalCond(re,env) );
   else if ( "lambda".equals(re.car()) ) return( evalLambda(re,env) );
   else if ( "rho".equals(re.car()) )    return( evalRho(re,env) );
   else return( apply(re,env) );  // function application
  }
 }

 /**\method"isSpecial(RPair)":
 Override to define new special forms (as "and")
 */
 boolean isSpecial(RPair p) { return(false); }

 /**\method"evalSpecial(RPair, REnvironment)":
 Override to define new special forms (as "and")
 */
 Object evalSpecial(RPair p, REnvironment env) { return(null); }

 /**\method"evalEval(RPair, REnvironment)"*/
 Object evalEval(RPair e, REnvironment env) {
  Object exp = eval(e.nth(1),env);
  Object xenv = eval(e.nth(2),env);
  if ( exp == null ) return(null);
  else if ( xenv == null) return( eval(exp,env) );
  else if ( env.getClass().isInstance(xenv) )
   return( eval(exp,(REnvironment)xenv) );
  else return(null);
 }

 /**\method"evalSet(RPair, REnvironment)"*/
 Object evalSet(RPair e, REnvironment env) {
  Object key = e.nth(1);
  if ( key == null ) return(null);
  else {
   Object val = eval(e.nth(2),env);
   return( env.set(key,val) );
  }
 }

 /**\method"evalDef(RPair, REnvironment)"*/
 Object evalDef(RPair e, REnvironment env) {
  Object key = e.nth(1);
  if ( key == null ) return(null);
  else {
   Object val = eval(e.nth(2),env);
   return( env.define(key,val) );
  }
 }

 /**\method"evalCond(RPair, REnvironment)"*/
 Object evalCond(RPair e, REnvironment env) {
  return( evalClauses(e.CDR(),env) );
 }
 private Object evalClauses(RPair cl, REnvironment env) {
  if (cl == null) return(null);
  Object car = cl.car();
  if ( car == null ) return(null);
  else {
   if ( !RPair.isRPair(car) || RPair.isNil(car) ) return(null);
   else {
    RPair first = (RPair)car;
    if ( isTrue(eval(first.car(),env)) )
     return(evalSequence(first,env));
    else return(evalClauses(cl.CDR(),env));
   }
  }
 }

 /**\method"evalSequence(RPair, REnvironment)"*/
 Object evalSequence(RPair e, REnvironment env) {
  if (e == null) return(null);
  if ( e.cdr() == null ) return( eval(e.car(),env) );
  else {
   eval(e.car(),env); // for side-effects
   return( evalSequence(e.CDR(),env) );
  }
 }

 /**\method"evalLambda(RPair, REnvironment)"*/
 Object evalLambda(RPair e, REnvironment env) {
  return( RPair.cons("LAMBDA", RPair.cons(env, e.CDR())) );
 }

 /**\method"evalRho(RPair, REnvironment)"*/
 Object evalRho(RPair e, REnvironment env) {
  return( RPair.cons("RHO", e.CDR()) );
 }

 /**\method"evalRPair(RPair, REnvironment)"*/
 RPair evalRPair(RPair e, REnvironment env) {
  if (e == null) return(null);
  if (e.isNil()) return(RPair.nil);
  else return( RPair.cons( eval(e.car(),env), evalRPair(e.CDR(),env) ) );
 }

 /**\method"apply(RPair, REnvironment)"

 The main applicator.

 It applies, where "l" is the list "(b c d)":
 {\medskipamount=0pt
 \point "(cons a l)" $\rightarrow$ "(a b c d)".
 \point "(car l)" $\rightarrow$ "b".
 \point "(cdr l)" $\rightarrow$ "(c d)".
 \point "(atom? exp)" $\rightarrow$ true $|$ false.
 \point "(eq? exp1 exp2)" $\rightarrow$ true $|$ false.
 }

 It accepts two kinds of compound aplication:
 {\medskipamount=0pt
 \point "((LAMBDA env formals body) actuals)" for functions.
 \point "((RHO name expander) expression)" for special forms.
 }
 */
 Object apply(RPair e, REnvironment env) {
  Object op = eval(e.car(),env);
  if ( op == null ) {
   System.err.println("ERROR: null function!");
   return(null);
  } else {
   if ( RPair.isRPair( op ) )
    return( applyCompound( (RPair)op,e.CDR(),env) );
   else if ( isPrimitive(op) )
    return( applyPrimitive(op, evalRPair(e.CDR(),env)) );
   else {
    System.err.println("ERROR: " + op + " undefined!");
    return(null);
   }
  }
 }

 /**\method"isPrimitive(Object)"*/
 boolean isPrimitive(Object op) {
  return( "cons".equals(op) || "car".equals(op) || "cdr".equals(op) ||
          "atom?".equals(op) || "eq?".equals(op) );
 }

 /**\method"applyPrimitive(Object, RPair)"*/
 Object applyPrimitive(Object op, RPair args) {
  Object first = args.nth(0);
  if ( "cons".equals(op) ) { Object second = args.nth(1);
   if ( RPair.isRPair(second) ) return( RPair.cons(first, (RPair)second) );
   else return(new RPair(first, second));
  } else if ( "car".equals(op) ) {
   if ( RPair.isRPair(first) ) return( ((RPair)first).car() );
   else return(null);
  } else if ( "cdr".equals(op) ) {
   if ( RPair.isRPair(first) ) return( ((RPair)first).Cdr() );
   else return(null);
  } else if ("atom?".equals(op)) {
   if ( RPair.isAtom( args.car() ) ) return(t);
   else return(RPair.nil);
  } else if ("eq?".equals(op)) { Object second = args.nth(1);
   if ( first == null )
    if ( second == null ) return(t); else return(RPair.nil);
   else
    if ( first.equals(second) ) return(t); else return(RPair.nil);
  } else return(null);
 }

 /**\method"applyCompound(RPair, RPair, REnvironment)"*/
 Object applyCompound(RPair op, RPair args, REnvironment env) {
  if ( "LAMBDA".equals(op.car()) ) return( applyLambda(op, args, env) );
  else if ( "RHO".equals(op.car()) ) return( applyRho(op, args, env) );
  else {
   System.err.println("ERROR: " + op.car() + " undefined!");
   return(null);
  }
 }

 /**\method"applyLambda(RPair, RPair, REnvironment)"

 It first evaluates the arguments in the current environment.
 Then it extends the stored environment, that in which the function
  was defined, with a new "RFrame" in which
  each formal argument is bound to its actual value.
 Finally the body is evaluated in the extended environment,
  alse known as closure.

 In applying "((lambda env (f0 f1) e0 ...) a0 a1)"  then
 "op" = "(lambda env (f0 f1) e0 ...)".
 */
 Object applyLambda(RPair op, RPair args, REnvironment currentenv) {
  Object result = null;
  try {
   RPair evargs = evalRPair(args, currentenv);
   REnvironment env = (REnvironment)(op.nth(1)); // stored env
   Object formals = op.nth(2); // (f0 f1)
   RPair body = op.CDR().CDR().CDR();  // (e0 ...)
   REnvironment ext = env.extend(new RFrame(formals,evargs));
   result = evalSequence( body, ext );
  } catch(Throwable t) { System.err.println("ERROR! in lambda: "+t); }
  return(result);
 }

 /**\method"applyRho(RPair, RPair, REnvironment)"

 It first reconstructs the expression, "cons"ing the "name".
 Then the reconstructed expression is "quote"d, that is, taken as data,.
  and, as such, is expanded by the "expander".
 Finally, the expanded expression is evaluated.

 In applying "((rho name expander) expression)"
 then "op" = "(rho name expander)",
 "(car (cdr op))" is the "name",
 and "(car (cdr (cdr op)))" is the "expander".
 */
 Object applyRho(RPair op, RPair args, REnvironment env) {
  Object result = null;
  try{
   Object name = op.CDR().car();
   Object expander = op.CDR().CDR().car();
   RPair expression = RPair.cons(name,args);
   RPair qexp = RPair.cons("quote", RPair.cons(expression,null));
   Object expansion = eval(RPair.cons(expander,RPair.cons(qexp,null)), env);
   result = eval(expansion, env);
  } catch(Throwable t) { System.err.println("ERROR! in rho: "+t); }
  return( result );
 }

 /**\method"toString()"*/
 public String toString() { return("RLisp"); }

}
