/** \class "RKeyboard"

An "RKeyboard" is a window to enter lines of text.

@author  Ramn Casares 2003
@version 2003.01.13
*/
package RLisp;

import java.awt.event.KeyListener;
import java.awt.event.KeyEvent;
import java.awt.event.ActionListener;
import java.awt.event.ActionEvent;

import javax.swing.JTextArea;
import javax.swing.JFrame;
import javax.swing.WindowConstants;
import javax.swing.text.BadLocationException;
import javax.swing.ScrollPaneConstants;
import javax.swing.JScrollPane;
import javax.swing.JButton;
import javax.swing.JToolBar;
import javax.swing.JLabel;
import javax.swing.Box;
import java.awt.Color;
import java.awt.Font;
import javax.swing.BoxLayout;
import java.awt.Container;

public class RKeyboard implements KeyListener, ActionListener {

 private JTextArea ta;
 private JLabel statuslabel;

 /** \variable"callingObject" */
 private ActionListener callingObject;

 /** \variable"endLine" is an invisible button
  that is clicked every time the carriage return is keyed */
 private RButton endLine;

 /** \variable"inputline" saves the last written line */
 private String inputline;

 /** \constructor "RKeyboard(ActionListener, String)" */
 RKeyboard(ActionListener callingObject, String firstline) {
  endLine = new RButton("Line", firstline);
  JFrame keyframe = new JFrame("Lisp from Keyboard");
  inputline = firstline;
  this.callingObject = callingObject;
  if (callingObject == null) {
   endLine.addActionListener(this);
   keyframe.setDefaultCloseOperation(WindowConstants.EXIT_ON_CLOSE);
  } else {
   endLine.addActionListener(callingObject);
   keyframe.setDefaultCloseOperation(WindowConstants.DISPOSE_ON_CLOSE);
  }
  keyframe.setBounds(200, 200, 500, 200);
  Container cp = keyframe.getContentPane();
  cp.setLayout(new BoxLayout(cp,BoxLayout.Y_AXIS));

  JToolBar toolBar = new JToolBar();
   JButton jbNesting = new JButton("Nesting");
    jbNesting.addActionListener(this);
    toolBar.add(jbNesting);
   JButton jbWord = new JButton("Word");
    jbWord.addActionListener(this);
    toolBar.add(jbWord);
   JButton jbMax = new JButton("Maximum");
    jbMax.addActionListener(this);
    toolBar.add(jbMax);
   JButton jbMin = new JButton("Minimum");
    jbMin.addActionListener(this);
    toolBar.add(jbMin);
   JButton jbNext = new JButton("Next");
    jbNext.addActionListener(this);
    toolBar.add(jbNext);
   JButton jbPre = new JButton("Previous");
    jbPre.addActionListener(this);
    toolBar.add(jbPre);
   cp.add(toolBar);

  ta = new JTextArea(15,40);
  ta.setEditable(true);
  ta.setLineWrap(false);
  ta.setBackground(new Color(0.5F,1.0F,0.5F));
  ta.setFont(new Font("Monospaced",Font.PLAIN,12));
  cp.add(new JScrollPane(ta,
   ScrollPaneConstants.VERTICAL_SCROLLBAR_AS_NEEDED,
   ScrollPaneConstants.HORIZONTAL_SCROLLBAR_AS_NEEDED));

  Box statusbox = Box.createHorizontalBox();
  statuslabel = new JLabel();
  statusbox.add(statuslabel);
  statusbox.add(Box.createHorizontalGlue());
  cp.add(statusbox);

  keyframe.setVisible(true);
  keyframe.pack();
  keyframe.show();
  ta.addKeyListener(this);
  ta.requestFocus();
 }

 public static int nesting(String text, int p) {
  if ( p > text.length() ) p = text.length();
  int l = 0;
  char c;
  for(int i=0; i<p; i++) { c = text.charAt(i);
   if (c == '(') l++; else if (c == ')') l--;
   if (l < 0) l = 0;
  }
  return(l);
 }

 public static int prePar(String text, int p, int q) {
  int pp = p;
  int qq = q;
  char c;
  while (qq>0 && pp>0) {
   c = text.charAt(--pp);
   if (c == ')') qq++; else if (c == '(') qq--;
  }
  while (pp>0 && (text.charAt(pp-1) == '\'')) pp--;
  return(pp);
 }

 public static int nextPar(String text, int p, int q) {
  int pp = p;
  int qq = q;
  char c;
  while (qq>0 && pp<text.length()) {
   c = text.charAt(pp++);
   if (c == ')') qq--; else if (c == '(') qq++;
  }
  return(pp);
 }

 private static char[] separator = " \t\n\r".toCharArray();

 public static String oneLine(String text) {
  while ( text.indexOf(';') != -1 ) {
   int l = text.length();
   int ini = text.indexOf(';');
   int fn = text.indexOf('\n',ini); if (fn==-1) fn = l;
   int fr = text.indexOf('\r',ini); if (fr==-1) fr = l;
   int fin = fn; if (fr<fn) fin = fr;
   text = text.substring(0,ini).concat(text.substring(fin,l));
  }
  for(int i=0; i<separator.length; i++)
   text = text.replace(separator[i], ' ');
  // text = text.replaceAll(" +"," "); // is Java 1.4
   int i = text.indexOf("  ");
   while (i >= 0) {
    text = (new StringBuffer(text)).replace(i,i+2," ").toString();
    i = text.indexOf("  ");
   }
  return(text);
 }

 public static boolean isInSet(char c, String set) {
  char[] cset = set.toCharArray();
  for(int i=0; i < cset.length; i++) if ( c == cset[i] ) return(true);
  return(false);
 }

 public static String thisWord(String text, int p) {
  if ( p < 0 ) p = 0;
  if ( p >= text.length() ) p = text.length() - 1;
  int ini = p;
  int fin = p;
  while ( ini > 0 &&
          !isInSet( text.charAt(ini-1)," \t\n\r\'()") ) ini--;
  while ( fin < text.length() &&
          !isInSet( text.charAt(fin)," \t\n\r\'()") ) fin++;
  return(text.substring(ini,fin));
 }

 String maxExpression(String text, int p){
  int nl = nesting(text,p);
  if (nl == 0) return(thisWord(text,p));
  else {
   int ini = prePar(text,p,nl);
   int fin = nextPar(text,p,nl);
   if ( ini < 0 || fin < 0 ) return("");
   else return(text.substring(ini,fin));
  }
 }

 public static String minExpression(String text, int p){
  int nl = nesting(text,p);
  if (nl == 0) return(thisWord(text,p));
  else {
   int ini = prePar(text,p,1);
   int fin = nextPar(text,p,1);
   if ( ini < 0 || fin < 0 ) return("");
   else return(text.substring(ini,fin));
  }
 }

 String preExpression(String text, int p){
  return(maxExpression(text,p-1));
 }

 String nextExpression(String text, int p){
  return(maxExpression(text,p+1)); }


 /** \method"keyTyped(KeyEvent e)"

 Implements the "KeyListener" interface.
 The only method that it is not empty is "keyTyped".
 */
 public void keyTyped(KeyEvent e) {
  char c = e.getKeyChar();
  if ( c == '\n' ) { // new line
   int cp = ta.getCaretPosition();
   int nl = nesting( ta.getText(), cp-1 );
   if ( nl == 0 ) {
    inputline = preExpression( ta.getText() , cp-1 );
    statuslabel.setText(inputline);
    endLine.setObject(oneLine(inputline));
    endLine.doClick();
   } else {
    statuslabel.setText("Nesting = " + nl);
    String ss = "";
    for(int i=0; i<nl; i++) ss = ss + " ";
    ta.insert(ss,cp);
   }
  } else if ( c == ')' ) {
   int cp = ta.getCaretPosition();
   String sta = ta.getText() + ")";
   int nl = nesting(sta, cp+1);
   statuslabel.setText("Nesting = " + nl);
  }
 }

 public void keyPressed(KeyEvent e) {}
 public void keyReleased(KeyEvent e) {}


 /**\method"actionPerformed(ActionEvent)"

 Implements the "ActionListener" interface.

 @param e the action event
 */
 public void actionPerformed(ActionEvent e) {
  boolean react = false;
  String texto = e.getActionCommand();
  if ("Line".equals(texto)) { react = false;
   inputline = ((RButton)(e.getSource())).getObject().toString();
  } else if ("Nesting".equals(texto)) { react = false;
   inputline = "Nesting = " + nesting(ta.getText(),ta.getCaretPosition());
  } else if ("Word".equals(texto)) { react = true;
   inputline = thisWord(ta.getText(),ta.getCaretPosition());
  } else if ("Maximum".equals(texto)) { react = true;
   inputline = maxExpression(ta.getText(),ta.getCaretPosition());
  } else if ("Minimum".equals(texto)) { react = true;
   inputline = minExpression(ta.getText(),ta.getCaretPosition());
  } else if ("Next".equals(texto)) { react = true;
   inputline = nextExpression(ta.getText(),ta.getCaretPosition());
  } else if ("Previous".equals(texto)) { react = true;
   inputline = preExpression(ta.getText(),ta.getCaretPosition());
  } else { react = false;
   System.err.println("ERROR: Action " + texto + " no implemented!");
  }
  statuslabel.setText(inputline);
  if ( callingObject == null || !react ) {
   System.out.println(oneLine(inputline));
  } else {
   endLine.setObject(oneLine(inputline));
   endLine.doClick();
  }
 }


 /** \method "main(String[])" to test the class

 @param args are the command line arguments
 */
 public static void main(String[] args) {
  RKeyboard k = new RKeyboard(null, "RConsole");
 }

}
