/** \class "RClassTree"

Class to select a constructor or a method from a tree. The leaves of the
tree are the constructors and methods defined in the classes that are
accesible from the selected directory or "jar" file.

@author  Ramn Casares 2002
@version 2002.08.05
*/
package RLisp;

import java.net.URL;

import java.lang.reflect.Array;
import java.lang.reflect.Field;
import java.lang.reflect.Constructor;
import java.lang.reflect.Method;

import java.io.File;
import java.util.Vector;
import java.util.Enumeration;

import java.util.jar.JarFile;
import java.util.jar.JarEntry;

import java.awt.Container;
import java.awt.Color;
import javax.swing.BoxLayout;
import javax.swing.JScrollPane;
import javax.swing.JFrame;
import javax.swing.WindowConstants;
import javax.swing.JButton;
import javax.swing.JToolBar;
import javax.swing.JTree;
import javax.swing.tree.DefaultMutableTreeNode;

import javax.swing.JFileChooser;

import java.awt.event.ActionListener;
import java.awt.event.ActionEvent;
import java.awt.event.WindowListener;
import java.awt.event.WindowEvent;
import javax.swing.event.TreeSelectionListener;
import javax.swing.event.TreeSelectionEvent;

public class RClassTree extends JFrame
 implements ActionListener, TreeSelectionListener {

 /**\variable"tree"*/
 private JTree tree;

 /**\variable"callingObject"
 is the "ActionListener" that receives the action events.
 */
 private ActionListener callingObject;

 /**\variable"ob" is the object selected so far.*/
 private Object ob;

 /**\variable"loader" to load classes from any directory */
 private RClassLoader loader;

 /**\variable"cd" is the selected directory or "jar" file */
 File cd;


 /**\constructor"RClassTree(ActionListener, boolean, RClassLoader)"

 Builds a "JFrame" console with a tree containing a branch for each
 class in the current directory. For each of these there are two branches,
 one for the class constructors and the other for the methods.

 @param callingObject is the "actionListener" object that will receive
                      the action events
 @param choosing to select a directory or just use the current one
 @param loader is the incremental "ClassLoader"
 */
 public RClassTree(ActionListener callingObject, boolean choosing,
  RClassLoader loader) {
  super("Class Tree");
  this.loader = loader;
  Container contentPane = this.getContentPane();
  contentPane.setLayout(new BoxLayout(contentPane,BoxLayout.Y_AXIS));

  this.callingObject = callingObject;
  if ( callingObject == null )
   this.setDefaultCloseOperation(WindowConstants.EXIT_ON_CLOSE);
  else
   this.setDefaultCloseOperation(WindowConstants.DISPOSE_ON_CLOSE);

  DefaultMutableTreeNode top = new DefaultMutableTreeNode("Classes");
  this.tree = new JTree(top);
  tree.addTreeSelectionListener(this);

  JToolBar toolBar = new JToolBar();
   RButton jbOK = new RButton("OK",this);
    if (callingObject == null) jbOK.addActionListener(this);
    else jbOK.addActionListener(callingObject);
    toolBar.add(jbOK);
    this.rootPane.setDefaultButton(jbOK);
   JButton jbName = new JButton("Name");
    if (callingObject == null) jbName.addActionListener(this);
    else jbName.addActionListener(callingObject);
    toolBar.add(jbName);
   //contentPane.add(toolBar,BorderLayout.NORTH);
   contentPane.add(toolBar);

  JScrollPane treeView = new JScrollPane(tree);
  treeView.setPreferredSize(new java.awt.Dimension(200, 200));

  String[] classes = getClasses(choosing);
  if ( classes == null || classes.length == 0 ) {
   if ( callingObject == null ) System.exit(0);
   else { this.dispose(); return; }
  }
  try{
   top.setUserObject( cd.getAbsolutePath() );
   DefaultMutableTreeNode mClass;
   DefaultMutableTreeNode mArray;
   DefaultMutableTreeNode mFields = null;
    Field[] fields = null;
   DefaultMutableTreeNode mNew = null;
    Constructor[] constructors = null;
   DefaultMutableTreeNode mMethods = null;
    Method[] methods = null;

   for (int i=0; i<classes.length; i++) {
    Class c = null;
    try { c = loader.loadClass(classes[i]);
    } catch (java.lang.NoClassDefFoundError ncdfe) {
     System.err.println("Class "+ ncdfe.getMessage() + " not loadable!");
     continue;
    }
    mClass = new DefaultMutableTreeNode(classes[i]);
    top.add(mClass);

    mArray = new DefaultMutableTreeNode("Array");
    mClass.add(mArray);
    mArray.add( new DefaultMutableTreeNode( Array.newInstance(c,1) ) );

    fields = c.getFields();
    if ( fields.length > 0 ) {
     mFields = new DefaultMutableTreeNode("Fields");
     mClass.add(mFields);
     for (int j=0; j<fields.length; j++) {
      mFields.add( new DefaultMutableTreeNode( fields[j] ) );
     }
    }
    constructors = c.getConstructors();
    if ( constructors.length > 0 ) {
     mNew = new DefaultMutableTreeNode("Constructors");
     mClass.add(mNew);
     for (int j=0; j<constructors.length; j++) {
      mNew.add( new DefaultMutableTreeNode( constructors[j] ) );
     }
    }
    methods = c.getMethods();
    if ( methods.length > 0 ) {
     mMethods = new DefaultMutableTreeNode("Methods");
     mClass.add(mMethods);
     for (int j=0; j<methods.length; j++) {
      mMethods.add( new DefaultMutableTreeNode( methods[j] ) );
     }
    }
   }
  } catch (Throwable t) { System.err.println(t); } //t.printStackTrace(); }
  contentPane.add(treeView);
  //tree.setBackground(new Color(1.0F,1.0F,0F)); // yellow
  this.setVisible(true);
  this.pack();
  this.show();
 }

 /**\constructor"RClassTree(boolean, RClassLoader)"*/
 public RClassTree(boolean choosing, RClassLoader loader) {
  this(null,choosing,loader);
 }

 /**\constructor"RClassTree(ActionListener, RClassLoader)"*/
 public RClassTree(ActionListener callingObject, RClassLoader loader) {
  this(callingObject,true,loader);
 }

 /**\constructor"RClassTree(RClassLoader)"*/
 public RClassTree(RClassLoader loader) { this(null,true,loader); }

 /**\method"getSelectedObject()"

 @return the selected object
 */
 public Object getSelectedObject() {
  return(ob);
 }

 /** \method"getClasses()"

 Returns an "array" with the names of all the classes having its
 ".class" file in the current directory.

 @return the "array" of names
 */
 private String[] getClasses() {
   cd = new File(System.getProperty("user.dir"));
   try { loader.addURL( cd.toURL() ); }
   catch(java.net.MalformedURLException mue) { System.err.println(mue); }
   RExtFilter efClass = new RExtFilter(".class");
   String[] oldlist = cd.list(efClass);
   int l;
   if (oldlist == null) l = 0; else l = oldlist.length;
   String[] newlist = new String[l];
   String aux;
   for (int i=0; i<l; i++) {
    aux = oldlist[i];
    newlist[i] = aux.substring(0,aux.lastIndexOf('.'));
   }
   return(newlist);
 }

 /** \method"getClasses(boolean)"

 If "choosing" is set to "true", then it returns an "array"
 with the names of all the classes having its ".class" file
 in the selected directory, recurring it, or "jar" file.

 If "choosing" is set to "false", then it returns an "array"
 with the names of all the classes having its ".class" file
 in the current directory.

 @param choosing should be true to select
 @return the "array" of names
 */
 private String[] getClasses(boolean choosing) {
  String[] acs = null;
  if (!choosing) return(getClasses());
  else {
   JFileChooser chooser = new JFileChooser();
   chooser.setFileSelectionMode(JFileChooser.FILES_AND_DIRECTORIES);
   chooser.setFileFilter(new RExtFilter(".jar"));
   chooser.setSelectedFile(new File(System.getProperty("user.dir")));
   int returnVal = chooser.showOpenDialog(null);
   if( returnVal == JFileChooser.APPROVE_OPTION) {
    cd = chooser.getSelectedFile();
    try {
     loader.addURL( cd.toURL() );
     System.err.println( cd.toURL().toString() );
     vector = new Vector();
     if ( cd.isDirectory() ) getAllClasses("",cd);
     else getAllJarClasses(new JarFile(cd));
     Object[] aco = vector.toArray();
     acs = new String[aco.length];
     for(int i=0; i<aco.length; i++) acs[i] = (String)aco[i];
    }
    catch(Throwable t) { System.err.println(t); }
    return(acs);
   } else return(null);
  }
 }

 private Vector vector;
 private void getAllClasses(String prefix, File file) {
  File[] dir = file.listFiles();
  String dirS = "";
  for(int i=0; i<dir.length; i++) { dirS = dir[i].getName();
   if ( dir[i].isDirectory() ) getAllClasses(prefix+dirS+".", dir[i]);
   else if ( dirS.lastIndexOf('.') >= 0 &&
    dirS.substring(dirS.lastIndexOf('.')).equals(".class") )
    vector.add( prefix + dirS.substring(0,dirS.lastIndexOf('.')) );
  }
 }
 private void getAllJarClasses(JarFile jf) {
  Enumeration jee = jf.entries();
  JarEntry je = null;
  String jen = null;
  while ( jee.hasMoreElements() ) {
   je = (JarEntry)jee.nextElement();
   jen = je.getName(); // System.out.println(jen);
   if ( !je.isDirectory() &&
    jen.lastIndexOf('.') >= 0 &&
    jen.substring(jen.lastIndexOf('.')).equals(".class") )
    vector.add( jen.substring(0,jen.lastIndexOf('.')).replace('/','.') );
  }
 }

 /**\method"actionPerformed(ActionEvent)"

 Implements the "ActionListener" interface.
 Usually the \See"RClassTree#callingObject" gets the action events and
 this implementation is only for testing purposes.

 @param e the action event
 */
 public void actionPerformed(ActionEvent e) {
  String texto = e.getActionCommand();
  if ("OK".equals(texto)) System.out.println(ob.toString());
  else if ("Name".equals(texto)) System.out.println("Name");
  else System.err.println("ERROR: Action " + texto + " no implemented!");
 }


 /**\method"valueChanged(TreeSelectionEvent)"

 Implements the "TreeSelectionListener" interface.
 Updates the \See"RClassTree#ob" object.

 @param e the tree selection event
 */
 public void valueChanged(TreeSelectionEvent e){
  DefaultMutableTreeNode node = (DefaultMutableTreeNode)
                                tree.getLastSelectedPathComponent();
  if (node == null) return;
  Object nodeInfo = node.getUserObject();
  if (node.isLeaf()) { ob = nodeInfo; }
 }


 /**\method"main(String[])"

 Just for testing this class.

 @param args command line options are ignored
 */
 public static void main(String[] args) {
  boolean choosing = true;
  if ( args.length > 0 && args[0].equals("false") ) choosing = false;
  URL[] urls = new URL[1];
  File ud = new File(System.getProperty("user.dir"));
  try { urls[0] = ud.toURL(); }
  catch (java.net.MalformedURLException mue) {} // always right
  RClassLoader rcl = new RClassLoader(urls);
  RClassTree frame = new RClassTree(null,choosing,rcl);
 }

}
